/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0);

// Freeze time or alow it to run according to taste. If time is frozen then
// do not freeze it at 0.0 as the shadows are then to intense. TheEmu.

float time = 0.7;               // Freeze time
// float time = iGlobalTime;    // Time allowed to run

float bubbleTime = iGlobalTime; // Always allow bubbles to move

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

#ifdef GL_ES
precision highp float;
#endif

// Rendering parameters
#define RAY_LENGTH_MAX     20.0
#define RAY_BOUNCE_MAX     10
#define RAY_STEP_MAX       80
#define LIGHT              vec3 ( 1.0, 1.0, -1.0 )
#define AMBIENT            0.2
#define SPECULAR_POWER     3.0
#define SPECULAR_INTENSITY 0.5

// Materials
struct Material {
   vec3 color;
   float behavior; // from -1.0 (fully reflective) to 1.0 (fully transparent)
   vec3 refractIndex; // not used if behavior < 0.0
};

Material getMaterial (in int index) {

   // Glass
   if (index == 0) {
      return Material (vec3 (0.8, 0.8, 0.8), 0.9, vec3 (1.50, 1.55, 1.60));
   }

   // Wine
   if (index == 1) {
      return Material (vec3 (1.0, 1.0, 0.0), 0.8, vec3 (1.50, 1.55, 1.60));
   }

   // Table
   return Material (vec3 (0.5, 0.5, 1.0), -0.5, vec3 (2.0, 2.1, 2.2));
}

// Math constants

#define DELTA  0.001
#define PI     3.14159265359

// Distance to the scene

float getDistance (in vec3 p, in int materialFilter, out int material) {
   p.y -= 1.0;
   const float deform = 2.2;
   float d = length (p * vec3 (deform, 1.0, deform));
   float dxz = length (p.xz);

   // Table
   float table = max (max (dxz - 1.6, p.y + 2.55), -p.y - 3.2);
   if ((materialFilter < 0 && table < 0.0) || materialFilter == 2) {
      material = 2;
      return table / deform;
   }

   // Champagne
   float wine = max (d - 1.4, p.y + p.x * 0.1 * cos (time * 2.0));
   wine = max (wine, 0.015 - length (mod (p - vec3 (0.0, bubbleTime, 0.0), 0.4) - 0.2));
   if ((materialFilter < 0 && wine < 0.0) || materialFilter == 1) {
      material = 1;
      return wine / deform;
   }

   // Glass
   float glass = max (max (d - 1.45, 1.4 - d), p.y - 0.4);
   glass = min (glass, max (dxz - 0.08, p.y + 1.4));
   glass = min (glass, max (dxz - 0.5, p.y + 2.5));
   glass = max (glass, -p.y - 2.55);
   if ((materialFilter < 0 && glass < 0.0) || materialFilter == 0) {
      material = 0;
      return glass / deform;
   }

   // Air
   material = -1;
   return min (min (wine, glass), table) / deform;
}

// Normal at a given point
vec3 getNormal (in vec3 p, in int material) {
   const vec2 h = vec2 (DELTA, -DELTA);
   return normalize (
      h.xxx * getDistance (p + h.xxx, material, material) +
      h.xyy * getDistance (p + h.xyy, material, material) +
      h.yxy * getDistance (p + h.yxy, material, material) +
      h.yyx * getDistance (p + h.yyx, material, material)
   );
}

// Cast a ray for a given color channel (and its corresponding refraction index)
vec3 lightDirection = normalize (LIGHT);
float raycast (in vec3 origin, in vec3 direction, in vec4 normal, in int materialTo, in float color, in vec3 channel) {

   // Check the behavior of the material
   Material material = getMaterial (materialTo);
   float alpha = abs (material.behavior);
   color *= 1.0 - alpha;

   // The ray continues...
   int materialFrom = -1;
   float refractIndexFrom = 1.0;
   for (int rayBounce = 1; rayBounce < RAY_BOUNCE_MAX; ++rayBounce) {

      // Interface with the material
      float refractIndexTo;
      vec3 refraction;
      if (materialTo == -1) {
         refractIndexTo = 1.0;
         refraction = refract (direction, normal.xyz, refractIndexFrom);
      } else {
         refractIndexTo = dot (material.refractIndex, channel);
         refraction = material.behavior < 0.0 ? vec3 (0.0) : refract (direction, normal.xyz, refractIndexFrom / refractIndexTo);
      }
      if (dot (refraction, refraction) < DELTA) {
         direction = reflect (direction, normal.xyz);
         origin += direction * DELTA * 2.0;
      } else {
         direction = refraction;
         materialFrom = materialTo;
         refractIndexFrom = refractIndexTo;
      }

      // Ray marching
      for (int rayStep = 0; rayStep < RAY_STEP_MAX; ++rayStep) {
         float dist = max (abs (getDistance (origin, -1, materialTo)), DELTA);
         normal.w += dist;
         if (materialFrom != materialTo || normal.w > RAY_LENGTH_MAX) {
            break;
         }
         origin += direction * dist;
      }

      // Check whether we hit something
      if (materialFrom == materialTo) {
         break;
      }

      // Get the normal
      if (materialTo == -1) {
         normal.xyz = -getNormal (origin, materialFrom);
      } else {
         normal.xyz = getNormal (origin, materialTo);

         // Basic lighting
         material = getMaterial (materialTo);
         float relfectionDiffuse = max (0.0, dot (normal.xyz, lightDirection));
         float relfectionSpecular = pow (max (0.0, dot (reflect (direction, normal.xyz), lightDirection)), SPECULAR_POWER) * SPECULAR_INTENSITY;
         float localColor = (AMBIENT + relfectionDiffuse) * dot (material.color, channel) + relfectionSpecular;
         float localAlpha = abs (material.behavior);
         color += localColor * (1.0 - localAlpha) * alpha;
         alpha *= localAlpha;
      }
   }

   // Get the background color
   float backColor = 1.0; // dot (textureCube (iChannel0, direction).rgb, channel);

   // Return the intensity of this color channel
   return color + backColor * alpha;
}

// Main function
void mainImage (out vec4 fragColor, in vec2 fragCoord) {

   // Define the ray corresponding to this fragment
   vec2 frag = (2.0 * fragCoord.xy - iResolution.xy) / iResolution.y;
   vec3 direction = normalize (vec3 (frag, 4.0));

   // Set the camera
   vec3 origin = 12.0 * vec3 ((cos (time * 0.1)), sin (time * 0.2), sin (time * 0.1));
   vec3 forward = -origin;
   vec3 up = vec3 (sin (time * 0.3), 2.0, 0.0);
   mat3 rotation;
   rotation [2] = normalize (forward);
   rotation [0] = normalize (cross (up, forward));
   rotation [1] = cross (rotation [2], rotation [0]);
   direction = rotation * direction;
    
   // Cast the initial ray
   vec4 normal = vec4 (0.0);
   int materialTo = -1;
   for (int rayStep = 0; rayStep < RAY_STEP_MAX; ++rayStep) {
      float dist = max (getDistance (origin, -1, materialTo), DELTA);
      normal.w += dist;
      if (materialTo != -1 || normal.w > RAY_LENGTH_MAX) {
         break;
      }
      origin += direction * dist;
   }

   // Check whether we hit something
   if (materialTo == -1) {
      fragColor.rgb = vec3(0.0); // textureCube (iChannel0, direction).rgb;
      fragColor.a = 0.0;
   } else {

      // Get the normal
      normal.xyz = getNormal (origin, materialTo);

      // Basic lighting
      float relfectionDiffuse = max (0.0, dot (normal.xyz, lightDirection));
      float relfectionSpecular = pow (max (0.0, dot (reflect (direction, normal.xyz), lightDirection)), SPECULAR_POWER) * SPECULAR_INTENSITY;
      fragColor.rgb = (AMBIENT + relfectionDiffuse) * getMaterial (materialTo).color + relfectionSpecular;

      // Cast a ray for each color channel
      fragColor.r = raycast (origin, direction, normal, materialTo, fragColor.r, vec3 (1.0, 0.0, 0.0));
      fragColor.g = raycast (origin, direction, normal, materialTo, fragColor.g, vec3 (0.0, 1.0, 0.0));
      fragColor.b = raycast (origin, direction, normal, materialTo, fragColor.b, vec3 (0.0, 0.0, 1.0));
      fragColor.a = 1.0;
   }

}

void main ( void )
 {
   mainImage ( gl_FragColor, gl_FragCoord.xy );
 }